package com.tencent.start.cgs.tools;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import static com.tencent.start.cgs.tools.App.*;

public class FileCopy {

    private static class CopyFailureException extends IOException {
        public CopyFailureException(Exception e) {
            super(e);
        }

        public CopyFailureException(String msg) {
            super(msg);
        }
    }

    public static void cp(String sourceUrl, String targetUrl, long maxReties)
            throws IOException {
        if (maxReties < 0) {
            maxReties = Integer.MAX_VALUE;
        }
        AbstractFile s = AbstractFile.make(sourceUrl);
        AbstractFile t = AbstractFile.make(targetUrl);
        if (t.isDirectory()) {
            t = t.getChildFile(AbstractFile.make(sourceUrl).getName());
        }
        targetUrl = t.url;
        String targetTempUrl = t.getParentFile().getChildFile(
                "~" + randomString(7) + "." + t.getName() + ".tmp").url;
        if (null == targetTempUrl) {
            throw new CopyFailureException("unknown");
        }
        long totalBytesWritten = 0;
        final long fileSize = s.length();
        final long traceIntervalNanoSeconds = 3000000000L;
        final String fileSizeString = humanReadableBytes(fileSize);
        s = t = null;
        App.nop(s, t);
        long failureCounter = 0;
        for (long i = 0; i < maxReties + 1 && totalBytesWritten < fileSize; ++i) {
            try {
                if (i > 0) {
                    long sleepMills = (1 == failureCounter ? 5 : (2 == failureCounter ? 15 : 30));
                    System.out.println("Wait for retry. (" + failureCounter + ", "+ sleepMills + "s)");
                    Thread.sleep(sleepMills * 1000);
                }
                AbstractFile source = AbstractFile.make(sourceUrl);
                AbstractFile targetTemp = AbstractFile.make(targetTempUrl);
                if (0 == totalBytesWritten) {
                    if (targetTemp.exists() && targetTemp.delete()) {
                        App.nop();
                    }
                } else if (totalBytesWritten > targetTemp.length()) {
                    throw new CopyFailureException("invalid file length: "
                            + totalBytesWritten + ", " + targetTemp.length());
                }
                try (InputStream in = source.getInputStream(totalBytesWritten);
                     OutputStream out = targetTemp.getOutputStream(totalBytesWritten)) {
                    if (0 == i) {
                        System.out.println("Source: " + sourceUrl + ", length " + humanReadableBytes(fileSize));
                    } else {
                        System.out.println("Transmission continued.");
                        System.out.println("Source: " + sourceUrl + ", length " + humanReadableBytes(fileSize)
                                + ", offset " + humanReadableBytes(totalBytesWritten));
                    }
                    System.out.println("Target: " + targetTempUrl);

                    long bytesWritten = 0;
                    final long startTimeNanoSeconds = System.nanoTime();
                    long nextTraceTimeNanoSeconds = startTimeNanoSeconds + traceIntervalNanoSeconds;
                    int bytes;
                    byte[] buffer = new byte[4 * 1024 * 1024];
                    while ((bytes = in.read(buffer)) > 0) {
                        out.write(buffer, 0, bytes);
                        failureCounter = 0;
                        bytesWritten += bytes;
                        totalBytesWritten += bytes;
                        long currentTimeNanoSeconds = System.nanoTime();
                        if (currentTimeNanoSeconds >= nextTraceTimeNanoSeconds) {
                            nextTraceTimeNanoSeconds = currentTimeNanoSeconds + traceIntervalNanoSeconds;
                            double speedMs = (double) bytesWritten * 1000000.0 /
                                    (currentTimeNanoSeconds - startTimeNanoSeconds);
                            System.out.println(
                                    humanReadableBytes(totalBytesWritten) + " / " + fileSizeString + ", "
                                            + String.format("%.2f", 100.0 * totalBytesWritten / fileSize) + '%' + ", "
                                            + humanReadableBytes(speedMs * 1000) + "/s, "
                                            + humanReadableTime((long) ((double) (fileSize - totalBytesWritten) / speedMs))
                                            + " remain.");
                        }
                    } //while ((bytes = in.read(buffer)) > 0)

                    if (totalBytesWritten != fileSize) {
                        throw new CopyFailureException("file length " + humanReadableBytes(fileSize)
                                + ", bytes written " + humanReadableBytes(totalBytesWritten));
                    }
                    final long totalTimeSpendNanoSeconds = System.nanoTime() - startTimeNanoSeconds;
                    final double speedMs = (double) bytesWritten * 1000000.0 / totalTimeSpendNanoSeconds;
                    System.out.println(fileSizeString + " / " + fileSizeString + ", 100%, "
                            + humanReadableBytes(speedMs * 1000) + "/s, "
                            + humanReadableTimeNanoSeconds(totalTimeSpendNanoSeconds) + ".");
                    System.out.println("Completed.");
                    AbstractFile target = AbstractFile.make(targetUrl);
                    if (target.exists() && target.delete()) {
                        App.nop();
                    }
                    if (!targetTemp.renameTo(target)) {
                        throw new CopyFailureException("count not move file \""
                                + targetTempUrl + "\" to \"" + targetUrl + "\"");
                    }
                    return;
                } // try (InputStream in; OutputStream out)
            } catch (CopyFailureException | InterruptedException e) {
                throw new CopyFailureException(e);
            } catch (Exception e) {
                ++failureCounter;
                System.err.println("Read/Write exception: " + e);
                //e.printStackTrace();
            }
        } // for (;;)
        throw new CopyFailureException("Copy failed.");
    }
}
